Hooks
-----
Hooks allow you to execute custom code at certain predefined points in the firmware execution. To use them, just define the hook function in your keymap file.

The following hooks are available available:

Hook function                   | Timing
--------------------------------|-----------------------------------------------
`void hook_early_init(void)`         | Early in the boot process, before the matrix is initialized and before a connection is made with the host. Thus, this hook has access to very few parameters, but it is a good place to define any custom parameters needed by other early processes.
`void hook_late_init(void)`          | Near the end of the boot process, after Boot Magic has run and LEDs have been initialized.
`void hook_bootmagic(void)`          | During the Boot Magic window, after EEPROM and Bootloader checks are made, but before any other built-in Boot Magic checks are made.
`void hook_usb_startup_wait_loop(void)` | Continuously, until the device gets ready and into USB configured state.
`void hook_usb_wakeup(void)`         | When the device wakes up from USB suspend state.
`void hook_usb_suspend_entry(void)`  | When the device enters USB suspend state.
`void hook_usb_suspend_loop(void)`   | Continuously, while the device is in USB suspend state. *Default action:* power down and periodically check the matrix, causing wakeup if needed.
`void hook_keyboard_loop(void)`      | Continuously, during the main loop, after the matrix is checked.
`void hook_matrix_change(keyevent_t event)`      | When a matrix state change is detected, before any other actions are processed.
`void hook_layer_change(uint32_t layer_state)`   | When any layer is changed.
`void hook_default_layer_change(uint32_t default_layer_state)`   | When any default layer is changed.
`void hook_keyboard_leds_change(uint8_t led_status)`             | Whenever a change in the LED status is performed. *Default action:* call `keyboard_set_leds(led_status)`
`bool hook_process_action(keyrecord_t *record)`                  | Before key event is processed by tmk_core. Return true if the event is consumed and default action is not needed.





### Hooks Examples

You can try these out by copying the code to your keymap file, or any .c file in the Makefile `SRC`.

#### Activate keymap layer 5 on startup

```C
#include "action_layer.h"

void hook_late_init(void)
{
    layer_on(5);
    print("Layer 5 enabled!");
}
```

#### Blink the Caps Lock LED every .5 seconds

```C
#include "timer.h"
#include "led.h"

bool my_led_status = 0;
uint16_t my_led_timer;

void hook_keyboard_loop(void)
{
    // check if we've reached 500 milliseconds yet...
    if (timer_elapsed(my_led_timer) > 500)
    {
        // we've reached 500 milliseconds!
        // reset the timer
        my_led_timer = timer_read();

        // check the current LED state
        if (my_led_status)
        {
            // LED is on, so let's turn it off
            led_set(host_keyboard_leds() & ~(1<<USB_LED_CAPS_LOCK));
            my_led_status = 0;
        }
        else
        {
            // LED is off, so let's turn it on
            led_set(host_keyboard_leds() | (1<<USB_LED_CAPS_LOCK));
            my_led_status = 1;
        }
    }
}
```

#### Flash the Caps Lock LED for 20ms on every keypress
```C
include "timer.h"
#include "led.h"

bool my_led_status = 0;
uint16_t my_led_timer;

void hook_matrix_change(keyevent_t event)
{
    // only flash LED for key press events, not key release events.
    if (event.pressed)
    {
        // check the current LED status and reverse it
        led_set(host_keyboard_leds() ^ (1<<USB_LED_CAPS_LOCK));

        my_led_status = 1;
        my_led_timer = timer_read();
    }
}

void hook_keyboard_loop(void)
{
    if (my_led_status)
    {
        // check if we've reached 20 milliseconds yet...
        if (timer_elapsed(my_led_timer) > 50)
        {
            led_set(host_keyboard_leds());

            my_led_status = 0;
        }
    }
}

```
